import pandas as pd
import numpy as np

def monthly_table(df: pd.DataFrame) -> pd.DataFrame:
    agg = df.groupby(df["date"].dt.to_period("M")).agg(
        mean_t_avg_c=("t_avg_c", "mean"),
        sum_rain_mm=("rain_mm", lambda s: s.dropna().sum()),
        mean_humidity=("humidity_pct", "mean"),
        valid_days=("date", "count"),
    ).reset_index(names="month")
    agg["month"] = agg["month"].astype(str)
    return agg

def _spells(series: pd.Series, cond) -> tuple[int, int]:
    ok = cond(series)
    blocks = (ok != ok.shift()).cumsum()
    lengths = ok.groupby(blocks).transform("size")
    maxlen = int(lengths.where(ok).max() or 0)
    count = int(len(lengths[ok].groupby(blocks[ok]).size()))
    return count, maxlen

def compute_all_stats(df: pd.DataFrame) -> dict:
    stats = {
        "coverage_days": int(df["date"].nunique()),
        "mean_t_avg_c": float(df["t_avg_c"].mean()),
        "t_min_c_min": float(df["t_min_c"].min()),
        "t_max_c_max": float(df["t_max_c"].max()),
        "mean_humidity": float(df["humidity_pct"].mean()),
        "rain_total_mm": float(df["rain_mm"].dropna().sum()),
        "hdd_base18": float(np.maximum(0, 18 - df["t_avg_c"]).sum()),
        "cdd_base22": float(np.maximum(0, df["t_avg_c"] - 22).sum()),
    }
    hw_count, hw_long = _spells(df["t_max_c"], lambda s: s >= 35)
    wet_count, wet_long = _spells(df["rain_mm"].fillna(0), lambda s: s >= 10)
    stats["heatwave_spells"] = hw_count
    stats["heatwave_longest_days"] = hw_long
    stats["wet_spells"] = wet_count
    stats["wet_longest_days"] = wet_long
    return stats