import pandas as pd

def validate_and_clean(df: pd.DataFrame) -> tuple[pd.DataFrame, dict]:
    qc = {"duplicates": 0, "imputed": {}, "excluded_rain_days": 0}
    # Drop duplicate dates and count them
    dedup = df.drop_duplicates(subset=["date"])
    qc["duplicates"] = int(len(df) - len(dedup))
    df = dedup.dropna(subset=["date"]).copy()

    # Range checks
    if "humidity_pct" in df:
        mask_bad_h = ~df["humidity_pct"].between(0, 100)
        df.loc[mask_bad_h, "humidity_pct"] = pd.NA

    if "rain_mm" in df:
        mask_bad_r = df["rain_mm"] < 0
        df.loc[mask_bad_r, "rain_mm"] = pd.NA

    if "t_min_c" in df and "t_max_c" in df:
        mask_bad_t = df["t_min_c"] > df["t_max_c"]
        swap = df.loc[mask_bad_t, ["t_min_c", "t_max_c"]].copy()
        df.loc[mask_bad_t, "t_min_c"] = swap["t_max_c"]
        df.loc[mask_bad_t, "t_max_c"] = swap["t_min_c"]

    df.sort_values("date", inplace=True)

    # Impute temps
    for col in ["t_min_c", "t_max_c"]:
        if col in df:
            n_missing = df[col].isna().sum()
            df[col] = df[col].fillna(method="ffill").fillna(method="bfill")
            qc["imputed"][col] = int(n_missing)

    # Humidity by monthly median
    if "humidity_pct" in df:
        df["month"] = df["date"].dt.month
        med = df.groupby("month")["humidity_pct"].transform("median")
        n_h = df["humidity_pct"].isna().sum()
        df["humidity_pct"] = df["humidity_pct"].fillna(med)
        qc["imputed"]["humidity_pct"] = int(n_h)

    # Rain rule
    if "rain_mm" in df:
        isna = df["rain_mm"].isna()
        run = (isna != isna.shift()).cumsum()
        lengths = isna.groupby(run).transform("size")
        long_gap = isna & (lengths > 2)
        qc["excluded_rain_days"] = int(long_gap.sum())
        df.loc[isna & ~long_gap, "rain_mm"] = 0.0

    # Derived
    if "t_min_c" in df and "t_max_c" in df:
        df["t_avg_c"] = (df["t_min_c"] + df["t_max_c"]) / 2
    else:
        df["t_avg_c"] = pd.NA
    df["week"] = df["date"].dt.isocalendar().week.astype(int)

    return df, qc