import pandas as pd
import matplotlib.pyplot as plt

def _save(fig, path):
    fig.tight_layout()
    fig.savefig(path, dpi=150)
    plt.close(fig)

def plot_daily_temps(df: pd.DataFrame, path):
    s = df.set_index("date")[["t_min_c","t_avg_c","t_max_c"]]
    roll = s.rolling(7, min_periods=1).mean().add_suffix("_r7")
    fig = plt.figure()
    s.plot(ax=plt.gca())
    roll.plot(ax=plt.gca(), linestyle="--", linewidth=1)
    plt.title("Daily Temperatures")
    plt.xlabel("Date"); plt.ylabel("°C")
    _save(fig, path)

def plot_monthly_rain(df: pd.DataFrame, path):
    m = df.groupby(df["date"].dt.to_period("M"))["rain_mm"].sum()
    fig = plt.figure()
    m.index = m.index.astype(str)
    m.plot(kind="bar")
    plt.title("Monthly Rainfall (mm)")
    plt.xlabel("Month"); plt.ylabel("mm")
    _save(fig, path)

def plot_boxplots(df: pd.DataFrame, path):
    df["month"] = df["date"].dt.month
    fig = plt.figure()
    df.boxplot(column="t_avg_c", by="month")
    plt.suptitle("")
    plt.title("Monthly Distribution of Average Temperature")
    plt.xlabel("Month"); plt.ylabel("°C")
    _save(fig, path)

def plot_calendar(df: pd.DataFrame, path):
    a = df.copy()
    a["dow"] = a["date"].dt.weekday
    a["week"] = a["date"].dt.isocalendar().week.astype(int)
    pivot = a.pivot_table(index="dow", columns="week", values="t_avg_c", aggfunc="mean")
    fig = plt.figure()
    plt.imshow(pivot, aspect="auto")
    plt.title("Weekly Heatmap of Avg Temp")
    plt.xlabel("ISO Week"); plt.ylabel("Day of Week (Mon=0)")
    plt.colorbar(label="°C")
    _save(fig, path)