from pathlib import Path
import json
import pandas as pd
import typer
from src.io_utils import load_weather
from src.quality import validate_and_clean
from src.stats import compute_all_stats, monthly_table
from src.plots import plot_daily_temps, plot_monthly_rain, plot_boxplots, plot_calendar

app = typer.Typer(help="Weather Data Analysis CLI")

@app.command()
def main(
    input: str = typer.Option(..., "--input", "-i", help="Path to input CSV"),
    outdir: str = typer.Option("out", "--outdir", "-o", help="Output directory"),
    year: int | None = typer.Option(None, "--year", "-y", help="Filter to a year"),
    fig_format: str = typer.Option("png", "--fig-format", help="png or pdf"),
    quiet: bool = typer.Option(False, "--quiet", help="Suppress console output")
):
    out = Path(outdir)
    (out / "figs").mkdir(parents=True, exist_ok=True)

    df = load_weather(input, year=year)
    df_clean, qc = validate_and_clean(df)
    df_clean.to_csv(out / "data_clean.csv", index=False)

    summary = compute_all_stats(df_clean)
    monthly = monthly_table(df_clean)
    monthly.to_csv(out / "monthly_stats.csv", index=False)
    (out / "summary.json").write_text(json.dumps(summary, indent=2))

    plot_daily_temps(df_clean, out / f"figs/daily_temps.{fig_format}")
    plot_monthly_rain(df_clean, out / f"figs/monthly_rain.{fig_format}")
    plot_boxplots(df_clean, out / f"figs/boxplots.{fig_format}")
    plot_calendar(df_clean, out / f"figs/calendar.{fig_format}")

    if not quiet:
        print(f"Rows in: {len(df)}  Rows clean: {len(df_clean)}")
        print("QC:", qc)

if __name__ == "__main__":
    app()